/* $Id: tachdelay.c,v 1.7 1999/06/03 21:42:59 dhiller Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* In the past, we have run into problems where a module with no
   active input channels has the wrong tach delay.  This program
   attempts to test that we don't have that problem. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "e1432.h"
#include "err1432.h"

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: tachdelay.c,v 1.7 1999/06/03 21:42:59 dhiller Exp $";
static const char *progname;

struct e1432_hwconfig hwconfig;

static int
init(SHORTSIZ16 laddr, E1432ID *hw, int *group, int *vibrato)
{
    SHORTSIZ16 chan;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    /* Force install, to root out uninitialized variable problems */
    CHECK(e1432_install(1, &laddr, 0, "/opt/e1432/lib/sema.bin"));
    CHECK(e1432_assign_channel_numbers(1, &laddr, hw));
    CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));

    *vibrato = hwconfig.model_code == E1432_MODEL_CODE_E1432;

    if (hwconfig.input_chans <= 0)
    {
	(void) fprintf(stderr, "%s: requires an input channel\n",
		       progname);
	return -1;
    }

    if (hwconfig.tach_chans > 0)
	chan = E1432_TACH_CHAN(1);
    else if (hwconfig.source_chans > 0)
	chan = E1432_SOURCE_CHAN(1);
    else
    {
	(void) fprintf(stderr, "%s: requires a source or tach channel\n",
		       progname);
	return -1;
    }

    *group = e1432_create_channel_group(*hw, 1, &chan);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

static int
get_delay(E1432ID hw, int group, int verbose, double *delay)
{
    FLOATSIZ32 d1, d2, diff;

    CHECK(e1432_init_measure(hw, group));
    CHECK(e1432_get_tach_delay(hw, group, &d1));
    if (verbose > 1)
	(void) printf("delay1 %g\n", d1);

    CHECK(e1432_reset_measure(hw, group));
    CHECK(e1432_get_tach_delay(hw, group, &d2));
    if (verbose > 2)
	(void) printf("delay2 %g\n", d2);

    diff = d1 - d2;
    if (d1 != 0)
	diff /= d1;
    if (diff < 0)
	diff = -diff;

    if (diff > 0.00001)
    {
	(void) fprintf(stderr,
		       "%s: delay change at reset_measure: %g -> %g\n",
		       progname, d1, d2);
	return -1;
    }

    if (d1 < -0.02 || d1 > 0.02)
    {
	(void) fprintf(stderr, "%s: delay unreasonable: %g\n",
		       progname, d1);
	return -1;
    }

    *delay = d1;
    return 0;
}

static int
compare(double d1, double d2)
{
    double diff;

    diff = d1 - d2;
    if (d1 != 0)
	diff /= d1;
    if (diff < 0)
	diff = -diff;

    if (diff > 0.00001)
    {
	(void) fprintf(stderr, "%s: delay changed: %g -> %g\n",
		       progname, d1, d2);
	return -1;
    }
    return 0;
}

static int
run(E1432ID hw, int group, int vibrato, int verbose)
{
    SHORTSIZ16 chan, error;
    int     igroup;
    double  tmp, d1, d2, d3, d4, d5, d6, d7, diff;
    FLOATSIZ32 span;

    if (verbose)
	(void) printf("After bootup, no input channels active\n");

    if (verbose)
	(void) printf("Span 20000\n");
    CHECK(get_delay(hw, group, verbose, &d1));

    if (verbose)
	(void) printf("Span 5000\n");
    CHECK(e1432_set_span(hw, group, 5000));
    CHECK(get_delay(hw, group, verbose, &d2));

    if (verbose)
	(void) printf("Span 20000\n");
    CHECK(e1432_set_span(hw, group, 20000));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d1, tmp));

    if (verbose)
	(void) printf("Span 2000\n");
    CHECK(e1432_set_span(hw, group, 2000));
    CHECK(get_delay(hw, group, verbose, &d3));

    if (verbose)
	(void) printf("Span 1250\n");
    CHECK(e1432_set_span(hw, group, 1250));
    CHECK(get_delay(hw, group, verbose, &d7));

    if (verbose)
	(void) printf("Span 1000\n");
    CHECK(e1432_set_span(hw, group, 1000));
    CHECK(get_delay(hw, group, verbose, &d6));

    if (verbose)
	(void) printf("Zoom on\n");
    CHECK(e1432_set_zoom(hw, group, E1432_ZOOM_ON));
    CHECK(get_delay(hw, group, verbose, &d4));

    if (verbose)
	(void) printf("Zoom off\n");
    CHECK(e1432_set_zoom(hw, group, E1432_ZOOM_OFF));
    CHECK(get_delay(hw, group, verbose, &tmp));
    /* On Vibrato, the span should now be rounded up to 2000.  On
       Sonata, the span should now be 1250. */
    CHECK(e1432_get_span(hw, group, &span));
    if (vibrato)
    {
	CHECK(compare(d3, tmp));

	diff = span - 2000;
	diff /= 2000;
	if (diff < 0)
	    diff = -diff;

	if (diff > 0.00001)
	{
	    (void) fprintf(stderr, "%s: expected span 2000, got %g\n",
			   progname, span);
	    return -1;
	}
    }
    else
    {
	CHECK(compare(d7, tmp));

	diff = span - 1250;
	diff /= 1250;
	if (diff < 0)
	    diff = -diff;

	if (diff > 0.00001)
	{
	    (void) fprintf(stderr, "%s: expected span 1250, got %g\n",
			   progname, span);
	    return -1;
	}
	/* Switch span to 2000 to match Vibrato */
	CHECK(e1432_set_span(hw, group, 2000));
    }

    if (verbose)
	(void) printf("Digital filter disabled\n");
    /* Because no input channels are enabled, this should be the same
       as filters enabled. */
    CHECK(e1432_set_anti_alias_digital(hw, E1432_INPUT_CHAN(1),
				       E1432_ANTI_ALIAS_DIGITAL_OFF));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Digital filter enabled\n");
    CHECK(e1432_set_anti_alias_digital(hw, E1432_INPUT_CHAN(1),
				       E1432_ANTI_ALIAS_DIGITAL_ON));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Data size I32\n");
    CHECK(e1432_set_data_size(hw, group, E1432_DATA_SIZE_32));
    CHECK(get_delay(hw, group, verbose, &tmp));
    if (!vibrato)
	/* Sonata returns a smaller value in 32-bit mode */
	tmp += 1.0 / 5120.0;
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Data size I16\n");
    CHECK(e1432_set_data_size(hw, group, E1432_DATA_SIZE_16));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Trigger Delay -5000\n");
    CHECK(e1432_set_trigger_delay(hw, group, -5000));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Trigger Delay 5000\n");
    CHECK(e1432_set_trigger_delay(hw, group, 5000));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Trigger Delay 0\n");
    CHECK(e1432_set_trigger_delay(hw, group, 0));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Oversampled\n");
    CHECK(e1432_set_decimation_oversample(hw, group,
					  E1432_DECIMATION_OVERSAMPLE_ON));
    CHECK(get_delay(hw, group, verbose, &tmp));
    if (!vibrato)
	/* Sonata returns a smaller value in oversample mode */
	tmp += 1.0 / 10240.0;
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Not oversampled\n");
    CHECK(e1432_set_decimation_oversample(hw, group,
					  E1432_DECIMATION_OVERSAMPLE_OFF));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("clock freq 65536\n");
    CHECK(e1432_set_clock_freq(hw, group, 65536));
    CHECK(e1432_set_span(hw, group, 6400));
    CHECK(get_delay(hw, group, verbose, &d5));

    if (verbose)
	(void) printf("clock freq 51200\n");
    CHECK(e1432_set_clock_freq(hw, group, 51200));
    CHECK(e1432_set_span(hw, group, 2000));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    chan = E1432_INPUT_CHAN(1);
    igroup = e1432_create_channel_group(hw, 1, &chan);
    if (igroup >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, igroup);
	return -1;
    }
    if (verbose)
	(void) printf("Now have an active input channel\n");

    if (verbose)
	(void) printf("Span 20000\n");
    CHECK(e1432_set_span(hw, group, 20000));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d1, tmp));

    if (verbose)
	(void) printf("Span 5000\n");
    CHECK(e1432_set_span(hw, group, 5000));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d2, tmp));

    if (verbose)
	(void) printf("Span 2000\n");
    CHECK(e1432_set_span(hw, group, 2000));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Span 1250\n");
    CHECK(e1432_set_span(hw, group, 1250));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d7, tmp));

    if (verbose)
	(void) printf("Span 1000\n");
    CHECK(e1432_set_span(hw, group, 1000));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d6, tmp));

    if (verbose)
	(void) printf("Zoom on\n");
    CHECK(e1432_set_zoom(hw, group, E1432_ZOOM_ON));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d4, tmp));

    if (verbose)
	(void) printf("Zoom off\n");
    CHECK(e1432_set_zoom(hw, group, E1432_ZOOM_OFF));
    CHECK(get_delay(hw, group, verbose, &tmp));
    /* On Vibrato, the span should now be rounded up to 2000.  On
       Sonata, the span should now be 1250. */
    CHECK(e1432_get_span(hw, group, &span));
    if (vibrato)
    {
	CHECK(compare(d3, tmp));

	diff = span - 2000;
	diff /= 2000;
	if (diff < 0)
	    diff = -diff;

	if (diff > 0.00001)
	{
	    (void) fprintf(stderr, "%s: expected span 2000, got %g\n",
			   progname, span);
	    return -1;
	}
    }
    else
    {
	CHECK(compare(d7, tmp));

	diff = span - 1250;
	diff /= 1250;
	if (diff < 0)
	    diff = -diff;

	if (diff > 0.00001)
	{
	    (void) fprintf(stderr, "%s: expected span 1250, got %g\n",
			   progname, span);
	    return -1;
	}
	/* Switch span to 2000 to match Vibrato */
	CHECK(e1432_set_span(hw, group, 2000));
    }

    if (verbose)
	(void) printf("Digital filter disabled\n");
    CHECK(e1432_set_anti_alias_digital(hw, E1432_INPUT_CHAN(1),
				       E1432_ANTI_ALIAS_DIGITAL_OFF));
    CHECK(get_delay(hw, group, verbose, &tmp));
    if (!vibrato)
    {
#if 0
	tmp += 1.0 / 5120.0;
	tmp -= 1.0 / 51200.0;
#else
	/* Sonata returns a larger? value when filters are disabled.
	   I really thought the above code worked, but now this code
	   is what works.  Not a big deal, I guess, unless it changes
	   back again... */
	tmp -= 1.0 / 5120.0;
	tmp += 1.0 / 51200.0;
#endif
    }
    CHECK(compare(d1, tmp));

    if (verbose)
	(void) printf("Digital filter enabled\n");
    CHECK(e1432_set_anti_alias_digital(hw, E1432_INPUT_CHAN(1),
				       E1432_ANTI_ALIAS_DIGITAL_ON));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Data size I32\n");
    CHECK(e1432_set_data_size(hw, group, E1432_DATA_SIZE_32));
    CHECK(get_delay(hw, group, verbose, &tmp));
    if (!vibrato)
	/* Sonata returns a smaller value in 32-bit mode */
	tmp += 1.0 / 5120.0;
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Data size I16\n");
    CHECK(e1432_set_data_size(hw, group, E1432_DATA_SIZE_16));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Trigger Delay -5000\n");
    CHECK(e1432_set_trigger_delay(hw, group, -5000));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Trigger Delay 5000\n");
    CHECK(e1432_set_trigger_delay(hw, group, 5000));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Trigger Delay 0\n");
    CHECK(e1432_set_trigger_delay(hw, group, 0));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Oversampled\n");
    CHECK(e1432_set_decimation_oversample(hw, group,
					  E1432_DECIMATION_OVERSAMPLE_ON));
    CHECK(get_delay(hw, group, verbose, &tmp));
    if (!vibrato)
	/* Sonata returns a smaller value in oversample mode */
	tmp += 1.0 / 10240.0;
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("Not oversampled\n");
    CHECK(e1432_set_decimation_oversample(hw, group,
					  E1432_DECIMATION_OVERSAMPLE_OFF));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    if (verbose)
	(void) printf("clock freq 65536\n");
    CHECK(e1432_set_clock_freq(hw, group, 65536));
    CHECK(e1432_set_span(hw, group, 6400));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d5, tmp));

    if (verbose)
	(void) printf("clock freq 51200\n");
    CHECK(e1432_set_clock_freq(hw, group, 51200));
    CHECK(e1432_set_span(hw, group, 2000));
    CHECK(get_delay(hw, group, verbose, &tmp));
    CHECK(compare(d3, tmp));

    /* This catches two former problems:
       1. init_measure would return "invalid decimation undersample"
          when it should have been "invalid span".
       2. After restoring zoom off, the next init measure could get a
          DMA error. */
    if (verbose)
	(void) printf("Zoom off after invalid zoom-on span\n");
    CHECK(e1432_set_span(hw, group, 20000));
    CHECK(e1432_set_zoom(hw, group, E1432_ZOOM_ON));
    CHECK(e1432_print_errors(0));
    error = e1432_init_measure(hw, group); /* Should error on Vibrato */
    CHECK(e1432_print_errors(1));
    if (error != ERR1432_ILLEGAL_SPAN)
    {
	/* This zoom span no problem on Sonata B. */
	if ( hwconfig.sca_id[0] == E1432_SCA_ID_SONATA_B )
	{
	    if ( error )
	    {
		(void) fprintf(stderr, "%s: Sonata B zoom returns %d\n",
		  progname, error);
	    }
	}
	else
	{
	    (void) fprintf(stderr, "%s: e1432_init_measure did not return "
	      "ERR1432_ILLEGAL_SPAN: %d\n", progname, error);
	    return -1;
	}
    }
    CHECK(e1432_reset_measure(hw, group));
    CHECK(e1432_set_zoom(hw, group, E1432_ZOOM_OFF));
    CHECK(get_delay(hw, group, verbose, &tmp));

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-uvV] [-L laddr]\n"
		   "\t-L: Logical address is <laddr>, default 8\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr;
    char   *p;
    int     c, vibrato, verbose, group;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    laddr = 8;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "L:uvV")) != -1)
	switch (c)
	{
	case 'L':
	    laddr = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr < 0 || laddr > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Run the measurement */
    if (init(laddr, &hw, &group, &vibrato) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, vibrato, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
